﻿/**
* jQuery FileAttach Plugin
*
* © TS Soft, 2012
* ts-soft.ru
*
* Requires: Modernizr, jquery.paste_image_reader
*/
; (function ($) {

    /* Usage:

        <form id="uploadForm" action="<YOUR_ACTION>">
            Attach files
            <input id="attachFile" type="file" name="files[]" />
            <div id="attachInputs" style="display:none;"></div>
            <div id="attachFileList"></div>
            <input type="button" onclick='$("#attachInputs").fileAttach("clear");' value='Clear' />
            <input type="button" onclick='alert("Files count: " + $("#attachInputs").fileAttach("totalCount"));' value='Count' />
        </form>
 
        <script type="text/javascript">
            $(document).ready(function () {
                $("#attachInputs").fileAttach({
                    dropTargets: "html",
                    pasteTargets: "html",
                    fileInput: "#attachFile",
                    inputsContainer: "#attachInputs",
                    fileListContainer: "#attachFileList",
                    form: "#uploadForm"
                });
            });
        </script>
 
        <style>
            .fileattach-dragover
            {
                background-color: #FFF8C1;
            }
            .fileattach-filename
            {
                font-family: Verdana;
            }
            .fileattach-filesize
            {
                font-family: Verdana;
                color: rgb(153, 153, 153);
            }
        </style>

    */

    var fileObjectsKey = 'fileAttach_fileObjects';
    var dataKey = 'fileAttach_data';

    $.fn.fileAttach = function(method) {

        var defaultOptions = {
            dropTargets: undefined,
            pasteTargets: undefined,
            fileInput: undefined,
            inputsContainer: undefined,
            fileListContainer: undefined,
            form: undefined,
            onSelectFile: undefined,
            onDropFile: undefined,
            onPasteFile: undefined
        };
        var sender = $(this);
        var options = defaultOptions;

        var methods = {
             init: function(newOptions) {
                options = $.extend(true, defaultOptions, newOptions);
                var data = sender.data(dataKey);
                if (!data) {
                    data = options;            
                    sender.data(dataKey, data);
                }

                if (Modernizr.draganddrop && !$.browser.msie 
                        && options.dropTargets) {
                    $(options.dropTargets).each(function() {
                        setupDropTargets($(this), onFileDrop);
                    });
                }
                if (Modernizr.draganddrop && !$.browser.msie 
                        && options.pasteTargets) {
                    $(options.pasteTargets).each(function() {
                        setupFilePasteTargets($(this), onFilePaste);
                    });
                }
                if (options.fileInput) {
                    $(options.fileInput).change(onFileSelect);
                }

                return sender;
            },
            clear: function (options) {
                var data = sender.data(dataKey);
                if (data) {
                    if (data.fileListContainer) {
                        $(data.fileListContainer).empty();
                    }
                    if (data.inputsContainer) {
                        $(data.inputsContainer).empty();
                    }
                    $(data.form).data(fileObjectsKey, new Array());
                }
                return sender;
            },
            totalCount: function (options) {
                var count = 0;
                var data = sender.data(dataKey);

                var inputs = $('input[type="file"]', $(data.inputsContainer));
                count += inputs.length;

                var fileObjects = $(data.form).data(fileObjectsKey);
                count += (fileObjects != undefined) ? fileObjects.length : 0;

                return count;
            }
        };

        if (methods[method]) {
            return methods[method].apply(this, Array.prototype.slice.call(arguments, 1));
        } else if (typeof method === 'object' || !method) {
            return methods.init.apply(this, arguments);
        } else {
            $.error('Method ' + method + ' does not exist on jQuery.fileAttach');
        }    

        function onFileDrop(dropSender, evt) {
            var files = evt.dataTransfer.files;
            for (var i = 0; i < files.length; i++) {
                attachFile(files[i], files[i].name);
            }

            if (options.onDropFile) {
                options.onDropFile(dropSender);
            }
        };

        function onFilePaste(data) {
            attachFile(data.blob, 'IMG_' + new Date().format('ddMMyyyy_HHmmss') + '.png');

            if (options.onPasteFile) {
                options.onPasteFile($(this));
            }
        };

        function onFileSelect() {
            $(this).blur();

            var fileSize = !$.browser.msie 
                ? (this.files[0].fileSize ? this.files[0].fileSize : this.files[0].size )
                : 0;
            cloneInput($(this), fileSize);

            if (options.onSelectFile) {
                options.onSelectFile($(this));
            }
        };

        function cloneInput(input, fileSize) {
            var newInput;
            if (!$.browser.opera) {
                newInput = input.clone(true);
                newInput.val("");
            } else {
                newInput = $('<input type="file" />');
                newInput.attr("name", input.attr("name"));
                newInput.change(onFileSelect);
            }
            
            newInput.insertAfter(input);
            input.appendTo(options.inputsContainer);

            addToList(input, fileSize);
        };

        function attachFile(data, caption) {
            var fileObjects = $(options.form).data(fileObjectsKey);
            if (fileObjects == undefined) {
                fileObjects = new Array();
            }
            fileObjects.push(data);
            $(options.form).data(fileObjectsKey, fileObjects);

            if (options.fileListContainer) {
                addFileHeader(caption, 'file', data, data.size);
            }
        };
    
        function addToList(input, fileSize) {
            if (options.fileListContainer) {
                addFileHeader(input.attr('value'), 'input', input, fileSize);
            }
        };

        function addFileHeader(caption, dataName, dataValue, fileSize) {
            caption = String(caption || '');
            var div = $('<div></div>');
            var spanCaption = $('<span class="fileattach-filename" title="' + caption + '">' + caption.match(/[^\/\\]+$/gi)[0] + '</span>');
            var spanHeader = $('<span style="cursor:pointer; text-decoration:underline;color:blue;">×</span>');
            var spanFileSize = $('<span class="fileattach-filesize">' + formatFileSize(fileSize) + '</span>');

            spanHeader.click(function() { detachFile(this); });
            spanHeader.data(dataName, dataValue);
            $(options.fileListContainer).append(div.append(spanHeader, ' ', spanCaption, ' ', spanFileSize));
        }

        function detachFile(sender) {
            sender = $(sender);

            var input = sender.data('input');
            if (input != undefined) {
                input.remove();
                sender.parent().remove();
                return;
            }
    
            var file = sender.data('file');
            if (file != undefined) {
                var files = $(options.form).data(fileObjectsKey);
                files.remove(file);
                sender.parent().remove();
                return;
            }
        };

        function formatFileSize(bytes)
        {
            var scale = 1024;
            var orders = [ "ГБ", "МБ", "КБ", "байт" ];

            var max = Math.pow(scale, orders.length - 1);
            var precision = 2;
            for (var i = 0; i < orders.length; i++)
            {
                if (bytes > max)
                {
                    return (bytes / max).toFixed(precision) + ' ' + orders[i];
                }
                max /= scale;
                if (precision > 0) {
                    precision--;
                }
            }
            return "";
        };
    
        function setupDropTargets(targets, onDrop) {
            if (!Modernizr.draganddrop || $.browser.msie) {
                return;
            }
            $(targets).each(function (index, item) {
                item.addEventListener("dragenter", onDragEnter, false);
                item.addEventListener("dragleave", onDragLeave, false);
                item.addEventListener("dragexit", noopHandler, false);
                item.addEventListener("dragover", noopHandler, false);
                item.addEventListener("drop", function (evt) { drop($(this), evt); onDrop($(this), evt); }, false);
            });
        };

        function onDragEnter(evt) {
            $(this).addClass('fileattach-dragover');
            return noopHandler(evt);
        };
        
        function onDragLeave(evt) {
            $(this).removeClass('fileattach-dragover');
            return noopHandler(evt);
        };
        
        function noopHandler(evt) {
            evt.stopPropagation();
            evt.preventDefault();
            return false;
        };
        
        function drop(sender, evt) {
            $(sender).removeClass('fileattach-dragover');
            evt.stopPropagation();
            evt.preventDefault();
        };

        function setupFilePasteTargets(selector, onPaste) {
            $(selector).pasteImageReader(function (data) {
                onPaste(data);
            });
        };
    };

    $.fn.getFileObjects = function(method) {
        var fileObjects = $(this).data(fileObjectsKey);
        return fileObjects != undefined ? fileObjects : new Array();
    };

})(jQuery);
